************************************************************************
*
*
*     Test problems for NonSmooth InEquality Constrained minimization
*
*
*     TNSIEC includes the following subroutines
*
*     S   XINIT3          Initiation of variables.
*     S   FUNC            Computation of the value and the subgradient 
*                           of the objective function.
*     S   CINEQ           Computation of the value and the subgradient 
*                           of the constraint functions.
*
*
*     Napsu Karmitsa (2003, inequality constrained version 2006-2007)      
*
*     Haarala M., Miettinen K. and Mkel M.M.: New Limited Memory
*     Bundle Method for Large-Scale Nonsmooth Optimization, Optimization
*     Methods and Software, Vol. 19, No. 6, 2004, 673-692.
*
*     Karmitsa N.: Test Problems for Large-Scale Nonsmooth Minimization,
*     Reports of the Department of Mathematical Information Technology,
*     Series B, Scientific Computing, B 4/2007, University of Jyvskyl, 
*     Jyvskyl, 2007.
*
*
************************************************************************
*      
*     * SUBROUTINE XINIT3 *
*
*      
*     * Purpose *
*
*     Initiation of variables for inequality constrained minimization.
*
*      
*     * Calling sequence *
*
*     CALL XINIT3(N,MG,X,NEXT,NCONS)
*
*
*     * Parameters *
*      
*     II  N          Number of variables.
*     IU  MG         Number of inequality constraints.
*     RO  X(N)       Vector of variables.
*     RI  NEXT       Problem number.
*     RI  NCONS      Constraint number.
*
*
*     * Problems *
*      
*     1.  Generalization of MAXQ.
*     2.  Generalization of MXHILB.
*     3.  Chained LQ.
*     4.  Chained CB3 I.
*     5.  Chained CB3 II.
*     6.  Number of active faces.
*     7.  Nonsmooth generalization of Brown function 2.
*     8.  Chained Mifflin 2. 
*     9.  Chained crescent I. 
*     10. Chained crescent II.
*
*
*     * Constraints *
*      
*     1.  Modification of Broyden tridiagonal constraints I for
*     problems 1,2,6,7,9, and 10.
*     1'. Modification of Broyden tridiagonal constraints I for 
*     problems 3,4,5, and 8.
*     2.  Modification of Broyden tridiagonal constraints II for 
*     problems 1,2,6,7,9, and 10.
*     2'. Modification of Broyden tridiagonal constraints II for 
*     problems 3,4,5, and 8.
*     3.  Modification of MAD1 I for problems 1 - 10.
*     4.  Modification of MAD1 II for problems 1 - 10.
*     5.  Simple modification of MAD1 I for problems 1,2,6,7,9, and 10.
*     5'. Simple modification of MAD1 I for problems 3,4,5, and 8.
*     6.  Simple modification of MAD1 II for problems 1,2,6,7,9, and 10.
*     6'. Simple modification of MAD1 II for problems 3,4,5, and 8.
*     7.  Modification of Problem 20 from UFO collection I for
*     problems 1 - 10.
*     8.  Modification of Problem 20 from UFO collection II for 
*     problems 1 - 10.
*
*
*     Napsu Karmitsa (2006-2007)      
*
*      
      
      SUBROUTINE XINIT3(N,MG,X,NEXT,NCONS)

*     Scalar Arguments
      INTEGER N,MG,NEXT,NCONS

*     Array Arguments
      DOUBLE PRECISION X(*)

*     Local Arguments
      INTEGER I,MGTMP

      MGTMP = MG

      GOTO (10,20,30,40,40,60,70,80,90,90) NEXT
      

      PRINT*,'Error: Not such a problem.'
      NEXT=-1
      RETURN

      
*    
*     Generalization of MAXQ
*

 10   CONTINUE

      DO 11 I=1,N/2
         X(I) = DBLE(I)
 11   CONTINUE
      DO 12 I=N/2+1,N
         X(I) = -DBLE(I)
 12   CONTINUE

      GOTO (13,17,14,15,19,16,13,17) NCONS


 13   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Broyden tridiagonal constraint I or P20 from UFO collection I:
c     NCONS = 1 or 7.

      IF (MG + 2 .GT. N/2) THEN
         DO 18 I=N/2+1,MG+2
            X(I) = DBLE(I)
 18      CONTINUE         
      END IF
      
      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      RETURN


 14   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 I. NCONS = 3.

      MG = 2

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF

      RETURN


 15   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 II. NCONS = 4.

      MG = 4

      X(1)=-0.50D+00
      X(2)=1.10D+00
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF

      RETURN


 16   CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 II. NCONS = 6.

      IF (MG .GT. N-1) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
         RETURN
      END IF

      DO 101 I=1,MG+1
         X(I) = 0.50D+00
 101  CONTINUE         
      
      RETURN


 17   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Broyden tridiagonal constraint II or  P20 from UFO collection II:
c     NCONS = 2 or 8.

      MG = 1
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 19   CONTINUE
c     Feasible starting point if the constraint is a simple 
c     modification of MAD1 I. NCONS = 5.

      MG = 1

      DO 111 I=1,N
         X(I) = 0.50D+00
 111  CONTINUE
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


      
*
*     Generalization of MXHILB
*      

 20   CONTINUE

      DO 21 I=1,N
         X(I) = 1.0D+00
 21   CONTINUE

      GOTO (23,29,25,26,121,27,24,28) NCONS


 23   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Broyden tridiagonal constraint I: NCONS = 1.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      RETURN


 24   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Problem 20 from UFO collection I: NCONS = 7.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 103 I=1,MG+2
         X(I) = 2.0D+00
 103  CONTINUE         
      
      RETURN

      
 25   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 I (for 2 constraints): NCONS = 3.

      MG = 2

      X(1)=-0.50D+00
      X(2)=1.10D+00
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN
      

 26   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 II (for 4 constraints): NCONS = 4.

      MG = 4

      X(1)=-0.50D+00
      X(2)=1.10D+00
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 27   CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 II: NCONS = 6.

      IF (MG .GT. N-1) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
         RETURN
      END IF

      DO 102 I=1,MG+1
         X(I) = 0.50D+00
 102  CONTINUE         

      RETURN


 28   CONTINUE
c     Feasible starting point if the constraint is a modification 
c     of Problem 20 from UFO collection II: NCONS = 8.

      MG = 1

      DO 22 I=1,N
         X(I) = 2.0D+00
 22   CONTINUE
         
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF

      RETURN


 29   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Broyden tridiagonal constraint II: NCONS = 2.

      MG = 1

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 121  CONTINUE
c     Feasible starting point if the constraint is a simple 
c     modification of MAD1 I: NCONS = 5.

      MG = 1

      DO 122 I=1,N
         X(I) = 0.50D+00
 122  CONTINUE
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


      
*     
*     Chained LQ
*     

 30   CONTINUE

      DO 31 I=1,N
         X(I)=-0.50D+00
 31   CONTINUE

      GOTO (33,38,35,36,37,39,34,38) NCONS


 33   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraints I: NCONS = 1.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 104 I=1,MG+2
         X(I) = 2.0D+00
 104  CONTINUE         
      
      RETURN

      
 34   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Problem 20 from UFO collection: NCONS = 7.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 105 I=1,MG+2
         X(I) = -2.0D+00
 105  CONTINUE         
      
      RETURN


 35   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 I: NCONS = 3.

      MG = 2

      X(1)=-0.50D+00
      X(2)=1.10D+00
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 36   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 II: NCONS = 4.

      MG = 4

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 37   CONTINUE
c     Feasible starting point if the constraint is a simple 
c     modification of MAD1 I: NCONS = 5.

      MG = 1
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 39   CONTINUE
c     Feasible starting point if the constraint is a simple 
c     modification of MAD1 II: NCONS = 6.

      IF (MG .GT. N-1) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      RETURN


 38   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraint II or Problem 20 from UFO 
c     collection II: NCONS = 2 or 8.
 
      MG = 1

      DO 32 I=1,N
         X(I) = 2.0D+00
 32   CONTINUE

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN



*     
*     Chained CB3 I and II
*

 40   CONTINUE
      
      DO 41 I=1,N
         X(I)=2.0D+00
 41   CONTINUE

      GOTO (42,46,43,44,47,45,42,46) NCONS

 42   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraint I or Problem 20 from UFO 
c     collection I: NCONS = 1 or 7.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF
      
      RETURN


 43   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 I: NCONS = 3.

      MG = 2

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 44   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 II: NCONS = 4.

      MG = 4

      X(1)=-0.50D+00
      X(2)=1.10D+00
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 45   CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 II: NCONS = 6.

      IF (MG .GT. N-1) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
         RETURN
      END IF

      DO 106 I=1,MG+1
         X(I) = 0.0D+00
 106  CONTINUE         

      RETURN
      

 46   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraint II or Problem 20 from UFO 
c     collection II: NCONS = 2 or 8.

      MG = 1
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN
      

 47   CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 I: NCONS = 5.

      MG = 1

      DO 48 I=1,N
         X(I)=0.0D+00
 48   CONTINUE

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN



*
*     Number of active faces
*      

 60   CONTINUE

      DO 61 I=1,N
         X(I) = 1.0D+00
 61   CONTINUE

      GOTO (63,69,65,66,161,67,64,68) NCONS

 63   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraint I: NCONS = 1.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF
      
      RETURN


 64   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Problem 20 from UFO collection I: NCONS = 7.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 107 I=1,MG+2
         X(I) = 2.0D+00
 107  CONTINUE         
      
      RETURN


 65   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 I: NCONS = 3.

      MG = 2

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 66   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 II: NCONS = 4.

      MG = 4

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 67   CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 II: NCONS = 6.

      IF (MG .GT. N-1) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 108 I=1,MG+1
         X(I) = 0.50D+00
 108  CONTINUE         

      RETURN


 68   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Problem 20 from UFO collection II: NCONS = 8.

      MG = 1
      DO 62 I=1,N
         X(I) = 2.0D+00
 62   CONTINUE

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 69   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraint II: NCONS = 2.

      MG = 1
      
      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 161  CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 I: NCONS = 5.

      MG = 1

      DO 162 I=1,N
         X(I) = 0.50D+00
 162   CONTINUE

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN



*      
*     Nonsmooth generalization of Brown function 2
*      

 70   CONTINUE

      DO 71 I=1,N
         IF(MOD(I,2) .EQ. 1) THEN
            X(I) = -1.0D+00
         ELSE
            X(I) = 1.0D+00
         ENDIF
 71   CONTINUE

      GOTO (73,79,75,76,171,77,74,78) NCONS


 73   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraints I: NCONS = 1.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 109 I=2,MG+2,2
         X(I) = -1.0D+00
 109  CONTINUE         
      
      RETURN


 74   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Problem 20 from UFO collection I: NCONS = 7.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 110 I=1,MG+2
         X(I) = 2.0D+00
 110  CONTINUE         

      RETURN


 75   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 I: NCONS = 3.

      MG = 2

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 76   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 II: NCONS = 4.

      MG = 4

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 77   CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 II: NCONS = 6.

      IF (MG .GT. N-1) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 112 I=1,MG+1
         X(I) = 0.50D+00
 112  CONTINUE         

      RETURN


 78   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Problem 20 from UFO collection II: NCONS = 8.

      MG = 1

      DO 72 I=1,N
         X(I) = 2.0D+00
 72   CONTINUE

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 79   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraints II: NCONS = 2.

      MG = 1

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 171  CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 I: NCONS = 5.

      MG = 1

      DO 172 I=1,N
         X(I) = 0.50D+00
 172   CONTINUE

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


      
*      
*     Chained Mifflin 2
*      

 80   CONTINUE

      DO 81 I=1,N
         X(I) = -1.0D+00
 81   CONTINUE

      GOTO (83,88,85,86,89,87,84,88) NCONS


 83   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraints I: NCONS = 1.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 113 I=1,MG+2
         X(I) = 2.0D+00
 113  CONTINUE         

      RETURN


 84   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Problem 20 from UFO collection I: NCONS = 7.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 114 I=1,MG+2
         X(I) = -2.0D+00
 114  CONTINUE         
      
      RETURN


 85   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 I: NCONS = 3.

      MG = 2

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 86   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 II: NCONS = 4.

      MG = 4

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 87   CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 II: NCONS = 6.

      IF (MG .GT. N-1) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 115 I=1,MG+1
         X(I) = 0.0D+00
 115  CONTINUE         
      
      RETURN


 88   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraint II or Problem 20 from UFO 
c     collection II: NCONS = 2 or 8.

      MG = 1

      DO 82 I=1,N
         X(I) = 2.0D+00
 82   CONTINUE

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 89   CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 I: NCONS = 5.

      MG = 1

      DO 181 I=1,N
         X(I) = 0.0D+00
 181  CONTINUE

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN



*      
*     Chained crescent I and II
*      

 90   CONTINUE

      DO 91 I=1,N
         IF(MOD(I,2) .EQ. 1) THEN
            X(I) = -1.50D+00
         ELSE
            X(I) = 2.0D+00
         ENDIF
 91   CONTINUE

      GOTO (92,97,94,95,98,96,93,97) NCONS


 92   CONTINUE
c     Feasible starting point if the constraint is a modification of 
c     Broyden tridiagonal constraints I: NCONS = 1.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 116 I=1,MG+2
         X(I) = -1.0D+00
 116  CONTINUE         
      
      RETURN


 93   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Problem 20 from UFO collection I: NCONS = 7.

      IF (MG .GT. N-2) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 117 I=1,MG+2
         X(I) = 2.0D+00
 117  CONTINUE         

      RETURN


 94   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 I: NCONS = 3.

      MG = 2

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 95   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     MAD1 II: NCONS = 4.

      MG = 4

      X(1)=-0.50D+00
      X(2)=1.10D+00

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


 96   CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 II: NCONS = 6.

      IF (MG .GT. N-1) THEN
         PRINT*,'Error: Too many constraints.'
         NEXT=-1
      END IF

      DO 118 I=1,MG+1
         X(I) = 0.50D+00
 118  CONTINUE         
      
      RETURN


 97   CONTINUE
c     Feasible starting point if the constraint is a modification of
c     Problem 20 from UFO collection II: NCONS = 8.

      MG = 1

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
    
      RETURN


 98   CONTINUE
c     Feasible starting point if the constraint is a simple modification
c     of MAD1 I: NCONS = 5.

      MG = 1

      DO 99 I=1,N
         X(I) = 0.50D+00
 99   CONTINUE

      IF (MG .GT. MGTMP) THEN
         PRINT*,'Error: Not enough space for constraints.'
         NEXT=-1
      END IF
      
      RETURN


      END

      
************************************************************************
*      
*     * SUBROUTINE FUNC *
*
*      
*     * Purpose *
*
*
*     Computation of the value and the subgradient of the objective
*     function.
*
*      
*     * Calling sequence *
*
*     CALL FUNC(N,X,F,G,NEXT)
*
*
*     * Parameters *
*      
*     II  N          Number of variables.
*     RI  X(N)       Vector of variables.
*     RI  NEXT       Problem number.
*     RO  F          Value of the objective function.
*     RO  G(N)       Subgradient of the objective function.
*
*
*     * Problems *
*      
*     1.  Generalization of MAXQ (convex).
*     2.  Generalization of MXHILB (convex).
*     3.  Chained LQ (convex).
*     4.  Chained CB3 I (convex).
*     5.  Chained CB3 II (convex).
*     6.  Number of active faces (nonconvex).
*     7.  Nonsmooth generalization of Brown function 2 (nonconvex).
*     8.  Chained Mifflin 2 (nonconvex). 
*     9.  Chained crescent I (nonconvex). 
*     10. Chained crescent II (nonconvex).
*     
*
*     Napsu Haarala (2003)      
*
*     
      
      SUBROUTINE FUNCI(N,X,F,G,NEXT)

*     Scalar Arguments
      INTEGER N,NEXT
      DOUBLE PRECISION F

*     Array Arguments
      DOUBLE PRECISION G(*),X(*)

*     Local Arguments
      INTEGER I,J,HIT
      DOUBLE PRECISION Y,TEMP2,TEMP3,A,B,C,D,P,Q

*     Intrinsic Functions
      INTRINSIC DABS,DMAX1,SIGN,DLOG,DEXP,DCOS,DSIN

      
      GOTO(10,20,30,40,50,60,70,80,90,100) NEXT

      
      PRINT*,'Error: Not such a problem.'
      NEXT=-1
      RETURN

      
*      
*     Generalization of MAXQ (convex)
*      
 10   CONTINUE
      F=X(1)*X(1)
      G(1)=0.0D+00
      HIT=1
      DO 11 I=2,N
         Y=X(I)*X(I)
         IF (Y .GT. F) THEN
            F=Y
            HIT=I
         END IF
         G(I)=0.0D+00
 11   CONTINUE

      G(HIT)=2*X(HIT)

      RETURN
      
      
*      
*     Generalization of MXHILB (convex)
*      
 20   CONTINUE
      F = 0.0D+00
      HIT=1
      DO 21 J = 1,N
         F = F + X(J)/DBLE(J)
 21   CONTINUE
      G(1)=SIGN(1.0D+00,F)
      F = DABS(F)
      DO 22 I = 2,N
         TEMP2 = 0.0D0
         DO 23 J = 1,N
            TEMP2 = TEMP2 + X(J)/DBLE(I+J-1)
 23      CONTINUE
         G(I)=SIGN(1.0D+00,TEMP2)
         TEMP2 = DABS(TEMP2)
         IF (TEMP2 .GT. F) THEN
            F=TEMP2
            HIT=I
         END IF
 22   CONTINUE
      TEMP3=G(HIT)
      DO 24 J = 1,N
         G(J) = TEMP3/DBLE(HIT+J-1)
 24   CONTINUE

      RETURN
      
      
*     
*     Chained LQ (convex)
*     
 30   CONTINUE
      F=0.0D+00
      G(1)=0.0D+00

      DO 31 I=1,N-1
         G(I+1)=0.0D+00
         A = -X(I)-X(I+1)
         B = -X(I)-X(I+1)+(X(I)*X(I)+X(I+1)*X(I+1)-1.0D+00)
         IF (A .GE. B) THEN
            F=F+A
            G(I)=G(I)-1.0D+00
            G(I+1)=-1.0D+00
         ELSE
            F=F+B
            G(I)=G(I)-1.0D+00+2.0D+00*X(I)
            G(I+1)=-1.0D+00+2.0D+00*X(I+1)
         ENDIF
 31   CONTINUE
      
      RETURN


*     
*     Chained CB3 (convex)
*     
 40   CONTINUE
      F=0.0D+00
      G(1)=0.0D+00

      DO 41 I=1,N-1
         G(I+1)=0.0D+00
         A=X(I)*X(I)*X(I)*X(I)+X(I+1)*X(I+1)
         B=(2.0D+00-X(I))*(2.0D+00-X(I))+
     &        (2.0D+00-X(I+1))*(2.0D+00-X(I+1))
         C= 2.0D+00*DEXP(-X(I)+X(I+1))
         Y=DMAX1(A,B)
         Y=DMAX1(Y,C)
         IF (Y .EQ. A) THEN
            G(I)=G(I)+4.0D+00*X(I)*X(I)*X(I)
            G(I+1)=2.0D+00*X(I+1)
         ELSE IF (Y .EQ. B) THEN
            G(I)=G(I)+2.0D+00*X(I)-4.0D+00
            G(I+1)=2.0D+00*X(I+1)-4.0D+00
         ELSE
            G(I)= G(I) - C
            G(I+1)= C
         END IF
         F=F+Y
 41   CONTINUE

      RETURN
      
      
*     
*     Chained CB3 2 (convex)
*
 50   CONTINUE
      F=0.0D+00
      G(1)=0.0D+00
      A=0.0D+00
      B=0.0D+00
      C=0.0D+00

      DO 51 I=1,N-1
         G(I+1)=0.0D+00
         A=A+X(I)*X(I)*X(I)*X(I)+X(I+1)*X(I+1)
         B=B+(2.0D+00-X(I))*(2.0D+00-X(I))+
     &        (2.0D+00-X(I+1))*(2.0D+00-X(I+1))
         C=C+2.0D+00*DEXP(-X(I)+X(I+1))
 51   CONTINUE
      F=DMAX1(A,B)
      F=DMAX1(F,C)
      IF (F .EQ. A) THEN
         DO 53 I=1,N-1
            G(I)=G(I)+4.0D+00*X(I)*X(I)*X(I)
            G(I+1)=2.0D+00*X(I+1)
 53      CONTINUE
      ELSE IF (F .EQ. B) THEN
         DO 54 I=1,N-1
            G(I)=G(I)+2.0D+00*X(I)-4.0D+00
            G(I+1)=2.0D+00*X(I+1)-4.0D+00
 54      CONTINUE
      ELSE
         DO 55 I=1,N-1
            G(I)= G(I) - 2.0D+00*DEXP(-X(I)+X(I+1))
            G(I+1)= 2.0D+00*DEXP(-X(I)+X(I+1))
 55      CONTINUE
      END IF

      RETURN
      
      
*      
*     Number of active faces (nonconvex)
*      
 60   CONTINUE

      TEMP3=1.0D+00
      Y=-X(1)
      G(1)= 0.0D+00
      F=DLOG(DABS(X(1))+1.0D+00)
      HIT=1
      TEMP2=F
      DO 62 I=2,N
         Y=Y - X(I)
         G(I)= 0.0D+00
         F=DMAX1(F,DLOG(DABS(X(I))+1.0D+00))
         IF(F .GT. TEMP2) THEN
            HIT=I
            TEMP2=F
         END IF
 62   CONTINUE
      F=DMAX1(F,DLOG(DABS(Y)+1.0D+00))
      IF(F .GT. TEMP2) THEN
         IF (Y.GE.0.0D+00) TEMP3=-1.0D+00
         DO 63 I=1,N
            G(I)= TEMP3*(1.0D+00/(DABS(Y)+1.0D+00))
 63      CONTINUE
      ELSE
         IF (X(HIT).LT.0.0D+00) TEMP3=-1.0D+00
         G(HIT)=TEMP3*(1.0D+00/(DABS(X(HIT))+1.0D+00))
      END IF
      
      RETURN
      
      
*
*     Nonsmooth generalization of Brown function 2 (nonconvex)
*      
 70   CONTINUE
      F=0.0D+00
      G(1)=0.0D+00
      DO 71 I=1,N-1
         A=DABS(X(I))
         B=DABS(X(I+1))
         C=X(I)*X(I)+1.0D+00
         D=X(I+1)*X(I+1)+1.0D+00
         F=F+B**C+A**D

         P=0.0D+00
         Q=0.0D+00
         IF (X(I).LT.0.0D+00) THEN
            IF (B .GT. P) P=DLOG(B)
            G(I)=G(I)-D*A**(D-1.0D+00)+2.0D+00*X(I)*P*B**C
         ELSE
            IF (B .GT. P) P=DLOG(B)
            G(I)=G(I)+D*A**(D-1.0D+00)+2.0D+00*X(I)*P*B**C
         ENDIF

         IF (X(I+1).EQ.0.0D+00) THEN
            G(I+1)=0.0D+00
         ELSE IF (X(I+1).LT.0.0D+00) THEN
            IF (A .GT. Q) Q=DLOG(A)
            G(I+1)=-C*B**(C-1.0D+00)+2.0D+00*X(I+1)*Q*A**D
         ELSE
            IF (A .GT. Q) Q=DLOG(A)
            G(I+1)=C*B**(C-1.0D+00)+2.0D+00*X(I+1)*Q*A**D
         ENDIF
 71   CONTINUE

      RETURN

      
*
*     Chained mifflin 2 (nonconvex)
*      
 80   CONTINUE
      F=0.0D+00
      G(1)=0.0D+00
      DO 81 I=1,N-1
         Y = X(I)*X(I) + X(I+1)*X(I+1) - 1.0D0
         F = F -X(I) + 2.0D+00*Y + 1.75D+00*DABS(Y)
         Y = SIGN(3.5D+00,Y) + 4.0D+00
         G(I) = G(I) + Y*X(I) - 1.0D+00
         G(I+1) = Y*X(I+1)
 81   CONTINUE
      
      RETURN

      
*
*     Chained crescent I (nonconvex)
*
 90   CONTINUE
      TEMP2=0.0D+00
      TEMP3=0.0D+00
      DO 91 I=1,N-1
         TEMP2 = TEMP2 + X(I)*X(I) + (X(I+1)-1.0D+00)*(X(I+1)-1.0D+00)
     &        + X(I+1) - 1.0D+00
         TEMP3 = TEMP3 - X(I)*X(I) - (X(I+1)-1.0D+00)*(X(I+1)-1.0D+00)
     &        + X(I+1) + 1.0D+00
 91   CONTINUE
      F = DMAX1(TEMP2,TEMP3)

      G(1)=0.0D+00
      IF (TEMP2 .GE. TEMP3) THEN
         DO 92 I=1,N-1
            G(I)=G(I)+2.0D+00*X(I)
            G(I+1)=2.0D+00*(X(I+1)-1.0D+00) + 1.0D+00
 92      CONTINUE
      ELSE
         DO 93 I=1,N-1
            G(I)=G(I)-2.0D+00*X(I)
            G(I+1)=-2.0D+00*(X(I+1)-1.0D+00) + 1.0D+00
 93      CONTINUE
      END IF
      
      RETURN


*     
*     Chained crescent II (nonconvex)
*
 100  CONTINUE
      F=0.0D+00
      G(1)=0.0D+00
      
      DO 101 I=1,N-1
         TEMP2 =  X(I)*X(I) + (X(I+1)-1.0D+00)*(X(I+1)-1.0D+00)
     &        + X(I+1) - 1.0D+00
         TEMP3 =  - X(I)*X(I) - (X(I+1)-1.0D+00)*(X(I+1)-1.0D+00)
     &        + X(I+1) + 1.0D+00
         IF (TEMP2 .GE. TEMP3) THEN
            F=F+TEMP2
            G(I)=G(I)+2.0D+00*X(I)
            G(I+1)=2.0D+00*(X(I+1)-1.0D+00) + 1.0D+00
         ELSE
            F=F+TEMP3
            G(I)=G(I)-2.0D+00*X(I)
            G(I+1)=-2.0D+00*(X(I+1)-1.0D+00) + 1.0D+00
         END IF
 101  CONTINUE

      RETURN
      
      END

      
************************************************************************
*      
*     * SUBROUTINE CINEQ *
*
*      
*     * Purpose *
*
*
*     Computation of the value and the subgradient of the constraint
*     functions.
*
*      
*     * Calling sequence *
*
*     CALL CINEQ(N,MG,X,G,DG,NEXT,NCONS)
*
*
*     * Parameters *
*      
*     II  N          Number of variables.
*     II  MG         Number of inequality constraints.
*     RI  X(N)       Vector of variables.
*     RI  NEXT       Problem number.
*     RI  NCONS      Constraint number.
*     RO  G(MG)      Values of the inequality constraints.
*     RO  DG(N*MG)   Jacobian of the inequality constraints.
*
*
*     * Problems *
*      
*     1.  Generalization of MAXQ.
*     2.  Generalization of MXHILB.
*     3.  Chained LQ.
*     4.  Chained CB3 I.
*     5.  Chained CB3 II.
*     6.  Number of active faces.
*     7.  Nonsmooth generalization of Brown function 2.
*     8.  Chained Mifflin 2. 
*     9.  Chained crescent I. 
*     10. Chained crescent II.
*
*
*     * Constraints *
*      
*     1.  Modification of Broyden tridiagonal constraints I for
*     problems 1,2,6,7,9, and 10.
*     1'. Modification of Broyden tridiagonal constraints I for 
*     problems 3,4,5, and 8.
*     2.  Modification of Broyden tridiagonal constraints II for 
*     problems 1,2,6,7,9, and 10.
*     2'. Modification of Broyden tridiagonal constraints II for 
*     problems 3,4,5, and 8.
*     3.  Modification of MAD1 I for problems 1 - 10.
*     4.  Modification of MAD1 II for problems 1 - 10.
*     5.  Simple modification of MAD1 I for problems 1,2,6,7,9, and 10.
*     5'. Simple modification of MAD1 I for problems 3,4,5, and 8.
*     6.  Simple modification of MAD1 II for problems 1,2,6,7,9, and 10.
*     6'. Simple modification of MAD1 II for problems 3,4,5, and 8.
*     7.  Modification of Problem 20 from UFO collection I for
*     problems 1 - 10.
*     8.  Modification of Problem 20 from UFO collection II for 
*     problems 1 - 10.
*
*
*     Napsu Karmitsa (2006-2007)      
*
*     
      
      SUBROUTINE CINEQ(N,MG,X,G,DG,NEXT,NCONS)

*     Scalar Arguments
      INTEGER N,MG,NEXT,NCONS

*     Array Arguments
      DOUBLE PRECISION X(*),G(*),DG(*)
c      DOUBLE PRECISION X(N),G(MG),DG(N*MG)

*     Local Arguments
      INTEGER I,HIT
      DOUBLE PRECISION TMP1

*     Intrinsic Functions
      INTRINSIC DSIN,DCOS



      DO 1 I = 1,N*MG
         DG(I) = 0.0D+00
 1    CONTINUE

      GOTO (10,90,40,50,100,60,30,80) NCONS

      
      PRINT*,'Error: Not such a problem.'
      NEXT=-1
      RETURN

*      
*     Modification of Broyden tridiagonal constraint I.
*

 10   CONTINUE

      GOTO(11,11,20,20,20,11,11,20,11,11) NEXT

 11   CONTINUE

c     MG <= N-2
c     for problems 1,2,6,7,9, and 10.

      DO 12 I = 1,MG
         G(I) = (3.0D+00-2.0D+00*X(I+1))*X(I+1) - X(I) - 2.0D+00*X(I+2)
     &        + 1.0D+00
         DG((I-1)*N+I)= DG((I-1)*N+I) - 1.0D+00
         DG((I-1)*N+I+1)= DG((I-1)*N+I+1) + 3.0D+00 - 4.0D+00*X(I+1)
         DG((I-1)*N+I+2)= DG((I-1)*N+I+2) - 2.0D+00
 12   CONTINUE

      RETURN

     
*     Modification of Broyden tridiagonal constraint I'.

 20   CONTINUE

c     MG <= N-2
c     for problems 3,4,5, and 8. 

      DO 21 I = 1,MG
         G(I) = (3.0D+00-2.0D+00*X(I+1))*X(I+1) - X(I) - 2.0D+00*X(I+2)
     &        + 2.50D+00
         DG((I-1)*N+I)= DG((I-1)*N+I) - 1.0D+00
         DG((I-1)*N+I+1)= DG((I-1)*N+I+1) + 3.0D+00 - 4.0D+00*X(I+1)
         DG((I-1)*N+I+2)= DG((I-1)*N+I+2) - 2.0D+00
 21   CONTINUE

      RETURN
      
      
*      
*     Modification of Problem 20 from UFO collection I.
*      
 30   CONTINUE

c     MG <= N-2

      DO 31 I = 1,MG
         G(I) = (3.0D+00-0.50D+00*X(I+1))*X(I+1) - X(I) - 2.0D+00*X(I+2)
     $        + 1.0D+00
         DG((I-1)*N+I)= DG((I-1)*N+I) - 1.0D+00
         DG((I-1)*N+I+1)= DG((I-1)*N+I+1) + 3.0D+00 - 1.0D+00*X(I+1)
         DG((I-1)*N+I+2)= DG((I-1)*N+I+2) - 2.0D+00
 31   CONTINUE

      RETURN
      
      
*     
*     Modification of MAD1 I.
*     
 40   CONTINUE

      G(1) = X(1)**2 + X(2)**2 + X(1)*X(2) - 1.0D+00
      HIT = 1

      TMP1 = DSIN(X(1))
      IF (TMP1 .GT. G(1)) THEN
         G(1) = TMP1
         HIT = 2
      END IF

      TMP1 = -DCOS(X(2))
      IF (TMP1 .GT. G(1)) THEN
         G(1) = TMP1
         HIT = 3
      END IF

      G(2) = -X(1) - X(2) + 0.5D+00
      DG(N+1) = -1.0D+00
      DG(N+2) = -1.0D+00

      GO TO (41,42,43) HIT
 41   CONTINUE
      DG(1) = 2.0D+00*X(1)+X(2)
      DG(2) = 2.0D+00*X(2)+X(1)

      RETURN

 42   CONTINUE
      DG(1) = DCOS(X(1))

      RETURN

 43   CONTINUE
      DG(2) = DSIN(X(2))

      RETURN


*     
*     Modification of MAD1 II.
*     
 50   CONTINUE

      G(1) = X(1)**2 + X(2)**2 + X(1)*X(2) - 1.0D+00
      DG(1) = 2.0D+00*X(1)+X(2)
      DG(2) = 2.0D+00*X(2)+X(1)

      G(2) = DSIN(X(1))
      DG(N+1) = DCOS(X(1))

      G(3) = -DCOS(X(2))
      DG(2*N+2) = DSIN(X(2))

      G(4) = -X(1) - X(2) + 0.5D+00
      DG(3*N+1) = -1.0D+00
      DG(3*N+2) = -1.0D+00

      RETURN
      
      
*     
*     Simple modification of MAD1 II.
*
 60   CONTINUE

      GOTO(70,70,61,61,61,70,70,61,70,70) NEXT

 61   CONTINUE

c     for problems 3,4,5, and 8.

      DO 62 I=1,MG
         G(I) = X(I)**2 + X(I+1)**2 + X(I)*X(I+1) - 1.0D+00
         DG((I-1)*N+I) = 2.0D+00*X(I)+X(I+1)
         DG((I-1)*N+I+1) = 2.0D+00*X(I+1)+X(I)
 62   CONTINUE

      RETURN
      
      
*     Simple modification of MAD1 II'.
      
 70   CONTINUE

c     for problems 1,2,6,7,9, and 10.

      DO 71 I=1,MG
         G(I) = X(I)**2 + X(I+1)**2 + X(I)*X(I+1)-2.0D+00*X(I)-
     &        2.0D+00*X(I+1)+1.0D+00
         DG((I-1)*N+I) = 2.0D+00*X(I)+X(I+1)-2.0D+00
         DG((I-1)*N+I+1) = 2.0D+00*X(I+1)+X(I)-2.0D+00
 71   CONTINUE

      RETURN

      
*      
*     Modification of Problem 20 from UFO collection II.
*      
 80   CONTINUE

      G(1) = (3.0D+00-0.50D+00*X(2))*X(2) - X(1) - 2.0D+00*X(3)
     &        + 1.0D+00
      DG(1)= - 1.0D+00
      DG(2)= 3.0D+00 - 1.0D+00*X(2)
      DG(3)= - 2.0D+00

      DO 81 I = 2,N-2
         G(1) = G(1)+(3.0D+00-0.50D+00*X(I+1))*X(I+1) - X(I) 
     &        - 2.0D+00*X(I+2) + 1.0D+00
         DG(I)= DG(I) - 1.0D+00
         DG(I+1)= DG(I+1) + 3.0D+00 - 1.0D+00*X(I+1)
         DG(I+2)= DG(I+2) - 2.0D+00
 81   CONTINUE

      
      RETURN

      
*      
*     Modification of Broyden tridiagonal constraint II.
*
 90   CONTINUE

      GOTO(91,91,93,93,93,91,91,93,91,91) NEXT

 91   CONTINUE

c     for problems 1,2,6,7,9, and 10.

      G(1) = (3.0D+00-2.0D+00*X(2))*X(2) - X(1) - 2.0D+00*X(3)
     &        + 1.0D+00
      DG(1)= - 1.0D+00
      DG(2)= 3.0D+00 - 4.0D+00*X(2)
      DG(3)= - 2.0D+00

      DO 92 I = 2,N-2
         G(1) = G(1)+(3.0D+00-2.0D+00*X(I+1))*X(I+1) - X(I) 
     &        - 2.0D+00*X(I+2) + 1.0D+00
         DG(I)= DG(I) - 1.0D+00
         DG(I+1)= DG(I+1) + 3.0D+00 - 4.0D+00*X(I+1)
         DG(I+2)= DG(I+2) - 2.0D+00
 92   CONTINUE

      RETURN

     
*     Modification of Broyden tridiagonal constraint II'.

 93   CONTINUE

c     for problems 3,4,5, and 8. 

      G(1) = (3.0D+00-2.0D+00*X(2))*X(2) - X(1) - 2.0D+00*X(3)
     &     + 2.50D+00
      DG(1)= - 1.0D+00
      DG(2)= 3.0D+00 - 4.0D+00*X(2)
      DG(3)= - 2.0D+00

      DO 94 I = 2,N-2
         G(1) = G(1)+(3.0D+00-2.0D+00*X(I+1))*X(I+1) - X(I) 
     &        - 2.0D+00*X(I+2) + 2.50D+00
         DG(I)= DG(I) - 1.0D+00
         DG(I+1)= DG(I+1) + 3.0D+00 - 4.0D+00*X(I+1)
         DG(I+2)= DG(I+2) - 2.0D+00
 94   CONTINUE

      RETURN


*     
*     Simple modification of MAD1 I.
*
 100  CONTINUE

      GOTO(110,110,111,111,111,110,110,111,110,110) NEXT

 111  CONTINUE

c     for problems 3,4,5, and 8.

      G(1) = X(1)**2 + X(2)**2 + X(1)*X(2) - 1.0D+00
      DG(1)= 2.0D+00*X(1) + X(2)
      DG(2)= 2.0D+00*X(2) + X(1)

      DO 112 I=2,N-1
         G(1) = G(1) + X(I)**2 + X(I+1)**2 + X(I)*X(I+1) - 1.0D+00
         DG(I) = DG(I) + 2.0D+00*X(I)+X(I+1)
         DG(I+1) = DG(I+1) + 2.0D+00*X(I+1)+X(I)
 112   CONTINUE

      RETURN
      
      
*     Simple modification of MAD1 I'.
      
 110   CONTINUE

c     for problems 1,2,6,7,9, and 10.

      G(1) = X(1)**2 + X(2)**2 + X(1)*X(2) - 2.0D+00*X(1) -
     &        2.0D+00*X(2) + 1.0D+00
      DG(1)= 2.0D+00*X(1) + X(2) - 2.0D+00
      DG(2)= 2.0D+00*X(2) + X(1) - 2.0D+00

      DO 113 I=2,N-1
         G(1) = G(1) + X(I)**2 + X(I+1)**2 + X(I)*X(I+1)-2.0D+00*X(I)-
     &        2.0D+00*X(I+1)+1.0D+00
         DG(I) = DG(I) + 2.0D+00*X(I)+X(I+1)-2.0D+00
         DG(I+1) = DG(I+1) + 2.0D+00*X(I+1)+X(I)-2.0D+00
 113  CONTINUE

      RETURN
      END
