from PyQt5.QtWidgets import *
from PyQt5.QtGui import *
from PyQt5.QtCore import *

from .editgroup import *
from .table import *
from . import values
from .delegates import QPComboBoxDelegate

__all__ = [ 'QPEditPostPlots' ]


class QPPostPlotsTableModel(QPGroupTableModel):
	def __init__(self, treePath, parent=None, *args):
		QPGroupTableModel.__init__(
			self, treePath, 
			header=[ "Plot name", "Title"], 
			editable=[ treePath.canRenameChildren(), True ], 
			dfl=treePath.childTemplate(), 
			parent=None, *args
		)
	
	# For display role the text is used, but otherwise codes are stored
	# Need to override these two because we are accessing stuff deeper in the data structure
	def data(self, index, role):
		if not index.isValid():
			return None
		elif role == Qt.DisplayRole or role == Qt.EditRole:
			row=index.row()
			col=index.column()
			if col==0:
				return QVariant(self.mylist[row][0])
			elif col==1:
				return QVariant(self.mylist[row][1]["title"])
			else:
				return None
		else:
			return None
	
	def setData(self, index, value, role):
		row=index.row()
		col=index.column()
		if col==0 and self.mylist[row][0]!=value:
			self.mylist[row][0]=value 
		elif col==1 and self.mylist[row][1]["title"]!=value:
			self.mylist[row][1]["title"]=value
		else:
			return False
		
		self.dataChanged.emit(index, index)
		return True

class QPStylesTableModel(QPTableModel):
	def __init__(self, data, parent=None, *args):
		QPTableModel.__init__(
			self, data, 
			header=[ 'Plot', 'Axes', 'Trace', 'Corner', 'Color', 'Line', 'Width', 'Symbol', 'Size' ], 
			dfl=values.blankStyle, 
			parent=None, *args
		)
	
	# For display role the text is used, but otherwise codes are stored
	# Need to override these two because we are accessing stuff deeper in the data structure
	def data(self, index, role):
		if not index.isValid():
			return None
		elif role == Qt.DisplayRole or role == Qt.EditRole:
			row=index.row()
			col=index.column()
			if col==5:
				if role == Qt.DisplayRole:
					return QVariant(values.lineStyleTranslator.toText(self.mylist[row][col]))
				else:
					return QVariant(self.mylist[row][col])
			if col==7:
				if role == Qt.DisplayRole:
					return QVariant(values.symbolStyleTranslator.toText(self.mylist[row][col]))
				else:
					return QVariant(self.mylist[row][col])
			elif col<9:
				return QVariant(self.mylist[row][col])
			else:
				return None
		else:
			return None
	
	def setData(self, index, value, role):
		row=index.row()
		col=index.column()
		if col>=0 and col<9 and self.mylist[row][col]!=value:
			# Validate (for pasted data)
			try:
				if col==5:
					values.lineStyleTranslator.toText(value)
				elif col==7:
					values.symbolStyleTranslator.toText(value)
			except Exception as e:
				return False
			
			self.mylist[row][col]=value 
		else:
			return False
		
		self.dataChanged.emit(index, index)
		return True

	
class QPEditPostPlots(QPEditGroup):
	def __init__(self, treePath=None, logger=None, parent=None, *args):
		QPEditGroup.__init__(self, QPPostPlotsTableModel, treePath, logger, parent=parent, *args)
		
		layout = QVBoxLayout(self)
		
		layout.setSpacing(4)
		# Layout should set the minimum and maximum size of the widget
		layout.setSizeConstraint(QLayout.SetMinAndMaxSize);
		
		self.tab=QPGroupTable(
			self.model, 
			canDelete=treePath.canDeleteChildren(), 
			canCreate=treePath.canCreateChildren(), 
			canMove=treePath.canMoveChildren(),
			buttons=False, 
			parent=self
		)
		
		self.stylesModel=QPStylesTableModel(self.data['styles'], parent=self)
		self.stylesWidget=QPTable(self.stylesModel, parent=self)

		self.stylesWidget.setItemDelegateForColumn(
			5, QPComboBoxDelegate(options=values.lineStyles, parent=self)
		)
		self.stylesWidget.setItemDelegateForColumn(
			7, QPComboBoxDelegate(options=values.symbolStyles, parent=self)
		)
		
		layout.addWidget(QLabel("Trace styles", self))
		layout.addWidget(self.stylesWidget)
		layout.addSpacing(2*layout.spacing())
		
		layout.addWidget(QLabel("Plots", self))
		layout.addWidget(self.tab)
		
		# Add a stretch at the bottom so that when member widgets shrink they are ordered at the top
		layout.addStretch(1)
		self.setLayout(layout)
		
		# Register model/view pairs
		self.registerModelView(self.model, self.tab)
		self.registerModelView(self.stylesModel, self.stylesWidget)
		
