"""
.. inheritance-diagram:: pyopus.evaluator.transform
    :parts: 1

Support for parameter transformation.
"""

__all__ = [ 'ParameterTransformation', 'TransformBase' ]


class TransformBase:
	"""
	Base class for transformation objects. 
	"""
	
	def __init__(self):
		pass
	

class ParameterTransformation:
	"""
	Performs transformations on parameter dictionaries.
	
	*transform* is a dictionary with parameter name as key holding 
	tranformation objects for parameters. If a parameter name 
	is not listed that parameter is not subject to trasformation. 
	
	*forwardMember* is the name of the transformation object's member 
	function that is used for the transformation of a parameter. 
	
	*inverseMember* is the name of the transformation object's member 
	function that is used for the inverse transformation of a parameter. 
	"""
	def __init__(self, transform, forwardMember, inverseMember):
		self.transform=transform
		self.forwardMember=forwardMember
		self.inverseMember=inverseMember
		self._compile()
	
	def _compile(self):
		self.fwd={}
		self.inv={}
		for name, xfrm in self.transform.items():
			self.fwd[name]=getattr(xfrm, self.forwardMember)
			self.inv[name]=getattr(xfrm, self.inverseMember)
		
	def __getstate__(self):
		state=self.__dict__.copy()
		del state['fwd']
		del state['inv']
		return state
	
	def __setstate__(self, state):
		self.__dict__.update(state)
		self._compile()
	
	def forwardTransform(self, params, inPlace=False):
		"""
		Performs a forward transformation. 
		"""
		names=list(params.keys())
		result={} if not inPlace else params
		for name in names:
			if name in self.fwd:
				result[name]=self.fwd[name](params[name])
		return result
	
	def inverseTransform(self, params, inPlace=False):
		"""
		Performs an inverse transformation. 
		"""
		names=list(params.keys())
		result={} if not inPlace else params
		for name in names:
			if name in self.inv:
				result[name]=self.inv[name](params[name])
		return result
	
	def __call__(self, params, inPlace=False, inverse=False):
		if inverse:
			return self.inverseTransform(params, inPlace)
		else:
			return self.forwardTransform(params, inPlace)
	
