#include "nomad.hpp"
using namespace std;
using namespace NOMAD;

const string DIABETES_FILE = "diabetes.txt";
const int    NB_X0_PTS     = 100;
const double EPS           = 1e-11;
const double PI            = 3.141592654;

// LH search used to generate a list of points:
bool LH_points ( int               n   ,
		 int               p   ,
		 const Point     & lb  ,
		 const Point     & ub  ,
		 vector<Point *> & pts   );

/*-------------------------------------*/
/*  class to represent one data point  */
/*-------------------------------------*/
class Data_Point {
private:
  int      _n;   // dimension
  double * _x;   // coordinates
  bool     _p;   // property

public:

  // constructor:
  Data_Point ( int n , const double * x , bool p ) : _n (n            ) ,
						     _x (new double[n]) ,
						     _p (p            )   {
    for ( int i = 0 ; i < _n ; ++i )
      _x[i] = x[i];
  }

  // destructor:
  ~Data_Point ( void ) { delete [] _x; }

  // get_p:
  bool get_p ( void ) const { return _p; }

  // dot product:
  double dot_product ( const double * x ) const;

  // display:
  void display ( const Display & out );

};

/*--------------------------------------*/
/*            custom evaluator          */
/*--------------------------------------*/
class My_Evaluator : public Evaluator {

private:

  int                          _n;    // dimension of x
  int                          _nm1;  // dimension of theta
  int                          _p;    // number of points
  const vector<Data_Point *> & _pts;  // the points

  int      _nh;    // histogram
  int    * _histo;

public:

  // constructor:
  My_Evaluator ( const Parameters           & param ,
		 int                          n     ,
		 int                          p     ,
		 const vector<Data_Point *> & pts     )
    : Evaluator ( param         ) ,
      _n        ( n             ) ,
      _nm1      ( n-1           ) ,
      _p        ( p             ) ,
      _pts      ( pts           ) ,
      _nh       ( 1 + _p/2      ) ,
      _histo    ( new int [_nh] )   {}

  // destructor:
  ~My_Evaluator ( void ) { delete [] _histo; }

  // eval_x:
  bool eval_x ( Eval_Point          & theta      ,
		const NOMAD::Double & h_max      ,
		bool                & count_eval   ) const;
};

/*------------------------------------------*/
/*                main function             */
/*------------------------------------------*/
int main ( int argc , char ** argv ) {

  // display:
  NOMAD::Display out ( std::cout );
  out.precision ( NOMAD::DISPLAY_PRECISION_STD );

  try {

    // NOMAD initializations:
    NOMAD::begin ( argc , argv );

    // read data (diabetes problem):
    vector<Data_Point *> pts;

    ifstream in ( DIABETES_FILE.c_str() );

    int    i , j;
    size_t k;

    int n; // dimension
    int p; // number of points

    {
      in >> n >> p;

      if ( in.fail() ) {
	cerr << "\ncannot read file " << DIABETES_FILE << "\n\n";
	in.close();
	return 1;
      }

      // out << "n=" << n << " p=" << p << endl;

      double * x = new double [n];
      bool     b;

      for ( j = 0 ; j < p ; ++j ) {
	for ( i = 0 ; i < n ; ++i )
	  in >> x[i];
	in >> b;
	pts.push_back ( new Data_Point ( n , x , b ) );
	// out << "point #" << pts.size()-1 << ": ";
	// pts[pts.size()-1]->display(out);
      }

      in.close();

      delete [] x;
    }

    int nm1 = n-1;

    // bounds:
    Point lb ( nm1 , 0.0  );
    Point ub ( nm1 , 2*PI );
    ub[0] = PI;

    // create list of starting points (based on LH):
    vector<Point *> x0_pts;

    LH_points ( nm1 , NB_X0_PTS , lb , ub , x0_pts );

    out << endl << open_block ( "starting points" );
    for ( k = 0 ; k < x0_pts.size() ; ++k )
      out << "#" << setw(2) << k << ":\t[ "
	  << *x0_pts[k] << " ]" << endl;
    out.close_block();

    // execute the tests:
    string test_names [4];
    test_names[0] = "Periodic Ortho (1/4)";
    test_names[1] = "Periodic GPS (2/4)";
    test_names[2] = "Aperiodic Ortho (3/4)";
    test_names[3] = "Aperiodic GPS (4/4)";
    //test_names[4] = "No bounds Ortho (5/6)";
    //test_names[5] = "No bounds GPS (6/6)";

    int res[4][NB_X0_PTS];

    int best_f = INT_MAX;

    for ( i = 0 ; i < 4 ; ++i ) {

      out << endl << open_block ( test_names[i] );

      for ( k = 0 ; k < x0_pts.size() ; ++k ) {

	out << k << " ";

	// parameters creation:
	Parameters param ( out );
	param.set_DIMENSION (nm1);
	param.set_DISPLAY_DEGREE (0);
	
	vector<bb_output_type> bbot (1);
	bbot[0] = NOMAD::OBJ;
	param.set_BB_OUTPUT_TYPE ( bbot );
	
	// param.set_MAX_BB_EVAL ( 50 );

	// starting point:
	param.set_X0 ( *x0_pts[k] );

	// periodic variables:
	if ( i < 2 ) {
	  ub[0] = 2*PI;
	  for ( j = 0 ; j < nm1 ; ++j )
	    param.set_PERIODIC_VARIABLE(j);
	}
	else
	  ub[0] = PI;

	// bounds:
	if ( i < 4 ) {
	  param.set_LOWER_BOUND ( lb );
	  param.set_UPPER_BOUND ( ub );
	}

	// GPS directions:
	if ( i%2 )
	  param.set_DIRECTION_TYPE ( GPS_2N_STATIC );

	param.set_F_TARGET ( 0 );
	
	// parameters check:
	param.check();
	
	// custom evaluator creation:
	My_Evaluator ev ( param , n , p , pts );
	
	// algorithm creation:
	Mads mads ( param , &ev );
	
	// MADS run:
	mads.run();

	// display and memorize result:
	const Eval_Point * x = mads.get_best_feasible();
	int f = ( x ) ? floor(x->get_f().value()) : INT_MAX;
	out << mads.get_stats().get_bb_eval() << " " << f << endl;
	res[i][k] = f;

	if ( f < best_f )
	  best_f = f;
      }
      
      out.close_block();
    }

    out << endl << "best f=" << best_f << endl;

    // delete starting points:
    for ( k = 0 ; k < x0_pts.size() ; ++k )
      delete x0_pts[k];

    // create the performance profiles:
    out << endl << open_block ( "performance profiles" );
    int    cnt;
    double alpha = 0.0;
    while ( alpha <= 100 ) {
      out << alpha << " ";
      for ( i = 0 ; i < 4 ; ++i ) {
	cnt = 0;
	for ( k = 0 ; k < x0_pts.size() ; ++k ) {
	  if ( (100.0 * (res[i][k] - best_f)) / best_f <= alpha )
	    ++cnt;
	}
	out << cnt << " ";
      }
      out << endl;
      alpha += 1.0;
    }
    out.close_block();

  }
  catch ( exception & e ) {
    cerr << "\nNOMAD has been interrupted (" << e.what() << ")\n\n";
  }

  NOMAD::Slave::stop_slaves ( out );
  NOMAD::end();

  return EXIT_SUCCESS;
}

/*-------------------------------*/
/*  methods of class Data_Point  */
/*-------------------------------*/

// dot product:
double Data_Point::dot_product ( const double * x ) const {
  double d = 0.0;
  for ( int i = 0 ; i < _n ; ++i )
    d += _x[i] * x[i];
  return d;
}

// display:
void Data_Point::display ( const Display & out ) {
  out << "x = ( ";
  for ( int i = 0 ; i < _n ; ++i )
    out << _x[i] << " ";
  out << ") p=" << _p << endl;
}

/*---------------------------------*/
/*  methods of class My_Evaluator  */
/*---------------------------------*/

// eval_x:
bool My_Evaluator::eval_x ( Eval_Point          & theta      ,
			    const NOMAD::Double & h_max      ,
			    bool                & count_eval   ) const {

  int      i;
  double * x = new double [_n];

  for ( i = 0 ; i < _nh ; ++i )
    _histo[i] = 0;

  // transform theta into x:
  // -----------------------
  double prod_sin = 1.0;
  for ( i = 0 ; i < _nm1 ; ++i ) {
    x[i] = prod_sin * cos(theta[i].value());
    if ( fabs(x[i]) < EPS )
      x[i] = 0.0;
    prod_sin *= sin(theta[i].value());
  }
  x[_nm1] = prod_sin;
  if ( fabs(x[_nm1]) < EPS )
    x[_nm1] = 0.0;

  // for ( i = 0 ; i < _n ; ++i )
  //  cout << "x[" << i << "]=" << x[i] << endl;

  // the objective function is the number of points that are misplaced:

  double rhs , dp;
  int    j , c1 , c2 , f_tmp , f = INT_MAX;
  bool   b;

  for ( i = 0 ; i < _p ; ++i ) {

    rhs = _pts[i]->dot_product ( x );

    // cout << "rhs=" << rhs << endl;

    c1 = c2 = 0;

    for ( j = 0 ; j < _p ; ++j ) {
      if ( i != j ) {

	dp = _pts[j]->dot_product ( x ) - rhs;
	b  = _pts[j]->get_p();
	
	if ( (b && dp < -EPS ) || (!b && dp > EPS ) )
	  ++c1;

	if ( (!b && dp < -EPS ) || (b && dp > EPS ) )
	  ++c2;

	// cout << "\tdp=" << dp << " p=" << b << endl;

      }
    }

    // cout << "\tc1=" << c1 << " c2=" << c2 << endl;

    f_tmp = (c1<c2) ? c1 : c2;

    ++_histo[f_tmp];

    if ( f_tmp < f )
      f = f_tmp;
  }

  delete [] x;

  count_eval = true;

  // compute the frac part from the histogram:
  // -----------------------------------------
  bool USE_FRAC = true;
  if ( USE_FRAC ) {

    double frac = 0.0 , old;

    int r = _n*_n , c;

    if ( r > _p )
      r = _p;

    double w = r;

    for ( i = f ; i < _nh ; ++i ) {

      c  = (r < _histo[i]) ? r : _histo[i];
      r -= c;

      old = frac;

      frac += static_cast<double>(c) / w;

      if ( r == 0 || frac - old < 1e-6 )
	break;

      w *= 2;
    }

    theta.set_bb_output ( 0 , f+1.0-frac );

  }

  else

    theta.set_bb_output ( 0 , f );

  return true;
}

/*---------------------------------------*/
/*  LH search used to generate p points  */
/*---------------------------------------*/
bool LH_points ( int                           n   ,
		 int                           p   ,
		 const NOMAD::Point          & lb  ,
		 const NOMAD::Point          & ub  ,
		 std::vector<NOMAD::Point *> & pts   ) {

  if ( n <= 0           ||
       p <= 0           ||
       !lb.is_defined() ||
       !ub.is_defined() ||
       lb.size() != n   ||
       ub.size() != n      )
    return false;

  for ( size_t j = 0 ; j < pts.size() ; ++j )
    delete pts[j];
  pts.clear();
  
  NOMAD::Point          * x;
  int                     i;
  int                     pm1 = p-1;
  NOMAD::Random_Pickup ** rps = new NOMAD::Random_Pickup *[n];

  for ( int k = 0 ; k < p ; ++k ) {
    x = new NOMAD::Point(n);  
    for ( i = 0 ; i < n ; ++i ) {
      if ( k==0 )
	rps[i] = new NOMAD::Random_Pickup(p);
      (*x)[i] = lb[i] +
	        (ub[i]-lb[i]) * ( rps[i]->pickup() + rand()/(1.0+D_INT_MAX)) / p;
      if ( k==pm1 )
	delete rps[i];
    }
    pts.push_back(x);
  }

  delete [] rps;
  
  return true;
}
